<?php
/**
 * Clase Auth - Manejo de autenticación y autorización
 */

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Iniciar sesión de usuario
     */
    public function login($username, $password) {
        $sql = "SELECT * FROM usuarios WHERE username = ? AND activo = 1";
        $user = $this->db->selectOne($sql, [$username]);
        
        if ($user && password_verify($password, $user['password'])) {
            // Regenerar ID de sesión por seguridad
            session_regenerate_id(true);
            
            // Guardar datos en sesión
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['nombre_completo'] = $user['nombre_completo'];
            $_SESSION['email'] = $user['email'];
            $_SESSION['cargo'] = $user['cargo'];
            $_SESSION['logged_in'] = true;
            $_SESSION['last_activity'] = time();
            
            // Actualizar último acceso
            $this->db->update('usuarios', 
                ['ultimo_acceso' => date('Y-m-d H:i:s')],
                'id = ?',
                [$user['id']]
            );
            
            // Registrar log
            $this->logActivity($user['id'], null, 'login', 'Inicio de sesión exitoso');
            
            return true;
        }
        
        // Registrar intento fallido
        $this->logActivity(null, null, 'login_failed', 'Intento de inicio de sesión fallido: ' . $username);
        
        return false;
    }
    
    /**
     * Cerrar sesión
     */
    public function logout() {
        if ($this->isLoggedIn()) {
            $this->logActivity($_SESSION['user_id'], null, 'logout', 'Cierre de sesión');
        }
        
        session_unset();
        session_destroy();
        
        // Regenerar ID de sesión
        session_start();
        session_regenerate_id(true);
    }
    
    /**
     * Verificar si el usuario está autenticado
     */
    public function isLoggedIn() {
        if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
            return false;
        }
        
        // Verificar timeout de sesión
        if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > SESSION_LIFETIME)) {
            $this->logout();
            return false;
        }
        
        // Actualizar tiempo de última actividad
        $_SESSION['last_activity'] = time();
        
        return true;
    }
    
    /**
     * Requerir autenticación
     */
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
                // Petición AJAX
                header('Content-Type: application/json');
                http_response_code(401);
                echo json_encode(['success' => false, 'message' => 'Sesión expirada', 'redirect' => 'login.php']);
                exit;
            } else {
                // Petición normal
                header('Location: login.php');
                exit;
            }
        }
    }
    
    /**
     * Obtener ID del usuario actual
     */
    public function getUserId() {
        return $_SESSION['user_id'] ?? null;
    }
    
    /**
     * Obtener datos del usuario actual
     */
    public function getUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['user_id'],
            'username' => $_SESSION['username'],
            'nombre_completo' => $_SESSION['nombre_completo'],
            'email' => $_SESSION['email'],
            'cargo' => $_SESSION['cargo']
        ];
    }
    
    /**
     * Cambiar contraseña
     */
    public function changePassword($userId, $newPassword) {
        $hashedPassword = password_hash($newPassword, PASSWORD_HASH_ALGO, ['cost' => PASSWORD_HASH_COST]);
        
        $updated = $this->db->update('usuarios',
            ['password' => $hashedPassword],
            'id = ?',
            [$userId]
        );
        
        if ($updated) {
            $this->logActivity($userId, null, 'change_password', 'Cambio de contraseña exitoso');
            return true;
        }
        
        return false;
    }
    
    /**
     * Registrar actividad en logs
     */
    public function logActivity($userId, $solicitudId, $accion, $descripcion) {
        try {
            $this->db->insert('logs_actividad', [
                'usuario_id' => $userId,
                'solicitud_id' => $solicitudId,
                'accion' => $accion,
                'descripcion' => $descripcion,
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
        } catch (Exception $e) {
            error_log("Error al registrar actividad: " . $e->getMessage());
        }
    }
    
    /**
     * Generar token CSRF
     */
    public function generateCsrfToken() {
        if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
        }
        return $_SESSION[CSRF_TOKEN_NAME];
    }
    
    /**
     * Validar token CSRF
     */
    public function validateCsrfToken($token) {
        return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }
}
