<?php
/**
 * API para manejo de solicitudes de certificados
 */

require_once '../../config/config.php';

header('Content-Type: application/json');

try {
    $solicitud = new Solicitud();
    $auth = new Auth();
    
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'POST':
            // Crear nueva solicitud
            handleCreate($solicitud, $auth);
            break;
            
        case 'GET':
            // Obtener solicitudes (requiere autenticación)
            $auth->requireLogin();
            handleGet($solicitud);
            break;
            
        case 'PUT':
            // Actualizar solicitud (requiere autenticación)
            $auth->requireLogin();
            handleUpdate($solicitud, $auth);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['success' => false, 'message' => 'Método no permitido']);
            break;
    }
    
} catch (Exception $e) {
    error_log("Error en API solicitudes: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error interno del servidor'
    ]);
}

/**
 * Crear nueva solicitud
 */
function handleCreate($solicitud, $auth) {
    // Validar datos requeridos
    $required = ['nombre_completo', 'rut', 'direccion', 'telefono', 'email', 'motivo'];
    foreach ($required as $field) {
        if (empty($_POST[$field])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => "El campo {$field} es requerido"
            ]);
            return;
        }
    }
    
    // Validar formato de email
    if (!filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'El formato del correo electrónico no es válido'
        ]);
        return;
    }
    
    // Validar reCAPTCHA en producción
    /*
    if (empty($_POST['g-recaptcha-response'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Por favor, complete el reCAPTCHA'
        ]);
        return;
    }
    
    $recaptcha = verificarRecaptcha($_POST['g-recaptcha-response']);
    if (!$recaptcha['success']) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Verificación reCAPTCHA fallida'
        ]);
        return;
    }
    */
    
    // Procesar archivos adjuntos
    $archivos = [];
    if (!empty($_FILES['adjuntos'])) {
        $archivos = procesarArchivos($_FILES['adjuntos']);
        if (isset($archivos['error'])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => $archivos['error']
            ]);
            return;
        }
    }
    
    // Crear solicitud
    $datos = [
        'nombre_completo' => trim($_POST['nombre_completo']),
        'rut' => trim($_POST['rut']),
        'direccion' => trim($_POST['direccion']),
        'telefono' => trim($_POST['telefono']),
        'email' => trim($_POST['email']),
        'motivo' => trim($_POST['motivo'])
    ];
    
    $resultado = $solicitud->crear($datos, $archivos);
    
    if ($resultado['success']) {
        // Enviar correo de confirmación
        try {
            $mailer = new Mailer();
            $mailer->enviarConfirmacionSolicitud($resultado['solicitud_id']);
        } catch (Exception $e) {
            error_log("Error al enviar correo de confirmación: " . $e->getMessage());
        }
        
        http_response_code(201);
    } else {
        http_response_code(400);
    }
    
    echo json_encode($resultado);
}

/**
 * Obtener solicitudes
 */
function handleGet($solicitud) {
    // Obtener parámetros de filtro
    $filtros = [
        'estado' => $_GET['estado'] ?? null,
        'busqueda' => $_GET['busqueda'] ?? null,
        'fecha_desde' => $_GET['fecha_desde'] ?? null,
        'fecha_hasta' => $_GET['fecha_hasta'] ?? null
    ];
    
    // Si se solicita una solicitud específica
    if (isset($_GET['id'])) {
        $datos = $solicitud->obtenerPorId($_GET['id']);
        
        if ($datos) {
            echo json_encode([
                'success' => true,
                'data' => $datos
            ]);
        } else {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Solicitud no encontrada'
            ]);
        }
        return;
    }
    
    // Obtener todas las solicitudes con filtros
    $datos = $solicitud->obtenerTodas($filtros);
    
    echo json_encode([
        'success' => true,
        'data' => $datos
    ]);
}

/**
 * Actualizar solicitud
 */
function handleUpdate($solicitud, $auth) {
    // Obtener datos PUT
    parse_str(file_get_contents("php://input"), $putData);
    
    if (empty($putData['id'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'ID de solicitud requerido'
        ]);
        return;
    }
    
    if (empty($putData['estado'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Estado requerido'
        ]);
        return;
    }
    
    $resultado = $solicitud->actualizarEstado(
        $putData['id'],
        $putData['estado'],
        $putData['observaciones'] ?? null
    );
    
    if ($resultado) {
        echo json_encode([
            'success' => true,
            'message' => 'Solicitud actualizada exitosamente'
        ]);
    } else {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Error al actualizar la solicitud'
        ]);
    }
}

/**
 * Procesar archivos adjuntos
 */
function procesarArchivos($files) {
    $archivos = [];
    $maxSize = MAX_FILE_SIZE;
    $allowedExtensions = ALLOWED_EXTENSIONS;
    
    // Reorganizar array de archivos
    if (isset($files['name']) && is_array($files['name'])) {
        $fileCount = count($files['name']);
        
        for ($i = 0; $i < $fileCount; $i++) {
            if ($files['error'][$i] === UPLOAD_ERR_OK) {
                $fileName = $files['name'][$i];
                $fileTmpName = $files['tmp_name'][$i];
                $fileSize = $files['size'][$i];
                $fileType = $files['type'][$i];
                
                // Validar tamaño
                if ($fileSize > $maxSize) {
                    return ['error' => "El archivo {$fileName} excede el tamaño máximo permitido"];
                }
                
                // Validar extensión
                $extension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                if (!in_array($extension, $allowedExtensions)) {
                    return ['error' => "El archivo {$fileName} tiene una extensión no permitida"];
                }
                
                $archivos[] = [
                    'name' => $fileName,
                    'tmp_name' => $fileTmpName,
                    'size' => $fileSize,
                    'type' => $fileType
                ];
            }
        }
    }
    
    return $archivos;
}

/**
 * Verificar reCAPTCHA
 */
function verificarRecaptcha($response) {
    $sql = "SELECT valor FROM configuracion WHERE clave = 'captcha_secret_key'";
    $db = Database::getInstance();
    $config = $db->selectOne($sql);
    
    $secretKey = $config['valor'];
    
    $verifyURL = 'https://www.google.com/recaptcha/api/siteverify';
    $data = [
        'secret' => $secretKey,
        'response' => $response,
        'remoteip' => $_SERVER['REMOTE_ADDR']
    ];
    
    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($data)
        ]
    ];
    
    $context = stream_context_create($options);
    $result = file_get_contents($verifyURL, false, $context);
    
    return json_decode($result, true);
}
