#!/bin/bash

###############################################################################
# Script de Instalación - Sistema de Certificados de Residencia
# Autor: Sistema Municipal
# Versión: 1.0
###############################################################################

echo "=========================================="
echo "Sistema de Certificados de Residencia"
echo "Instalación Automática"
echo "=========================================="
echo ""

# Colores
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Verificar si el script se ejecuta como root
if [ "$EUID" -ne 0 ]; then 
    echo -e "${RED}Por favor, ejecute como root (sudo)${NC}"
    exit 1
fi

# Verificar requisitos del sistema
echo -e "${YELLOW}Verificando requisitos del sistema...${NC}"

# PHP
if ! command -v php &> /dev/null; then
    echo -e "${RED}PHP no está instalado${NC}"
    echo "Instalando PHP..."
    apt-get update
    apt-get install -y php8.1 php8.1-cli php8.1-mysql php8.1-mbstring php8.1-xml php8.1-gd php8.1-zip php8.1-curl
fi

# MySQL
if ! command -v mysql &> /dev/null; then
    echo -e "${RED}MySQL no está instalado${NC}"
    echo "Instalando MySQL..."
    apt-get install -y mysql-server
fi

# Composer
if ! command -v composer &> /dev/null; then
    echo -e "${RED}Composer no está instalado${NC}"
    echo "Instalando Composer..."
    curl -sS https://getcomposer.org/installer | php
    mv composer.phar /usr/local/bin/composer
    chmod +x /usr/local/bin/composer
fi

echo -e "${GREEN}✓ Requisitos verificados${NC}"
echo ""

# Solicitar información de configuración
echo "Configuración de la Base de Datos:"
read -p "Host de MySQL [localhost]: " DB_HOST
DB_HOST=${DB_HOST:-localhost}

read -p "Nombre de la base de datos [certificados_residencia]: " DB_NAME
DB_NAME=${DB_NAME:-certificados_residencia}

read -p "Usuario de MySQL [root]: " DB_USER
DB_USER=${DB_USER:-root}

read -sp "Contraseña de MySQL: " DB_PASS
echo ""

# Crear base de datos
echo -e "${YELLOW}Creando base de datos...${NC}"
mysql -h"$DB_HOST" -u"$DB_USER" -p"$DB_PASS" < database/schema.sql

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Base de datos creada exitosamente${NC}"
else
    echo -e "${RED}✗ Error al crear la base de datos${NC}"
    exit 1
fi

# Instalar dependencias
echo -e "${YELLOW}Instalando dependencias de Composer...${NC}"
composer install --no-dev --optimize-autoloader

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Dependencias instaladas${NC}"
else
    echo -e "${RED}✗ Error al instalar dependencias${NC}"
    exit 1
fi

# Crear directorios necesarios
echo -e "${YELLOW}Creando directorios...${NC}"
mkdir -p uploads/solicitudes
mkdir -p uploads/certificados
mkdir -p certificates
mkdir -p logs

# Crear archivos .gitkeep
touch uploads/solicitudes/.gitkeep
touch uploads/certificados/.gitkeep
touch certificates/.gitkeep
touch logs/.gitkeep

# Configurar permisos
echo -e "${YELLOW}Configurando permisos...${NC}"
chmod -R 755 uploads/
chmod -R 755 certificates/
chmod -R 777 uploads/solicitudes/
chmod -R 777 uploads/certificados/
chmod -R 777 logs/

echo -e "${GREEN}✓ Permisos configurados${NC}"

# Actualizar archivo de configuración
echo -e "${YELLOW}Actualizando configuración...${NC}"

cat > config/config.php << EOF
<?php
/**
 * Archivo de configuración principal del sistema
 */

// Configuración de la base de datos
define('DB_HOST', '$DB_HOST');
define('DB_NAME', '$DB_NAME');
define('DB_USER', '$DB_USER');
define('DB_PASS', '$DB_PASS');
define('DB_CHARSET', 'utf8mb4');

// Configuración de la aplicación
define('APP_NAME', 'Sistema de Certificados de Residencia');
define('APP_VERSION', '1.0.0');
define('APP_URL', 'http://localhost/certificados-residencia');

// Configuración de directorios
define('BASE_PATH', dirname(__DIR__));
define('PUBLIC_PATH', BASE_PATH . '/public');
define('UPLOAD_PATH', BASE_PATH . '/uploads');
define('SOLICITUDES_PATH', UPLOAD_PATH . '/solicitudes');
define('CERTIFICADOS_PATH', UPLOAD_PATH . '/certificados');
define('CERTIFICATES_PATH', BASE_PATH . '/certificates');

// Configuración de archivos
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_EXTENSIONS', ['pdf', 'jpg', 'jpeg', 'png']);
define('MAX_FILES_PER_REQUEST', 5);

// Configuración de sesión
define('SESSION_NAME', 'CERTRES_SESSION');
define('SESSION_LIFETIME', 3600); // 1 hora

// Configuración de seguridad
define('CSRF_TOKEN_NAME', 'csrf_token');
define('PASSWORD_HASH_ALGO', PASSWORD_BCRYPT);
define('PASSWORD_HASH_COST', 10);

// Configuración de zona horaria
date_default_timezone_set('America/Santiago');

// Configuración de errores (cambiar en producción)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', BASE_PATH . '/logs/php_errors.log');

// Configuración de PHP
ini_set('upload_max_filesize', '10M');
ini_set('post_max_size', '10M');
ini_set('max_execution_time', '300');
ini_set('memory_limit', '256M');

// Headers de seguridad
header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

// Autoload de clases
spl_autoload_register(function (\$class) {
    \$file = BASE_PATH . '/src/' . str_replace('\\\\', '/', \$class) . '.php';
    if (file_exists(\$file)) {
        require_once \$file;
    }
});

// Iniciar sesión
if (session_status() === PHP_SESSION_NONE) {
    session_name(SESSION_NAME);
    session_start([
        'cookie_lifetime' => SESSION_LIFETIME,
        'cookie_httponly' => true,
        'cookie_secure' => false,
        'cookie_samesite' => 'Strict'
    ]);
}
EOF

echo -e "${GREEN}✓ Configuración actualizada${NC}"

# Resumen
echo ""
echo "=========================================="
echo -e "${GREEN}Instalación Completada Exitosamente${NC}"
echo "=========================================="
echo ""
echo "Credenciales de acceso por defecto:"
echo "Usuario: admin"
echo "Contraseña: admin123"
echo ""
echo -e "${YELLOW}⚠️  IMPORTANTE: Cambie la contraseña después del primer acceso${NC}"
echo ""
echo "URLs del sistema:"
echo "Formulario público: http://localhost/certificados-residencia/public/"
echo "Panel de administración: http://localhost/certificados-residencia/public/login.php"
echo ""
echo "Próximos pasos:"
echo "1. Configurar SMTP en el panel de administración"
echo "2. Configurar Google reCAPTCHA"
echo "3. Personalizar textos y logos según necesidad"
echo ""
echo "Para soporte: soporte@ejemplo.com"
echo "=========================================="
