# 🔧 Solución: Error al Generar Certificado

## ❌ Problema
Al hacer clic en "Generar Certificado" aparece el error: **"Error al comunicarse con el servidor"**

---

## ✅ SOLUCIONES

### Solución 1: Instalar Dependencias de Composer (Recomendada)

El error ocurre porque faltan librerías PHP para generar PDFs.

```bash
cd /ruta/certificados-residencia
composer require tecnickcom/tcpdf
composer require dompdf/dompdf
```

Si Composer no está instalado:
```bash
# Instalar Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Luego instalar dependencias
cd /ruta/certificados-residencia
composer install
```

---

### Solución 2: Verificar Permisos de Directorios

```bash
cd /ruta/certificados-residencia
chmod -R 777 uploads/
chmod -R 777 uploads/certificados/
mkdir -p uploads/certificados/$(date +%Y)/$(date +%m)
chmod -R 777 uploads/certificados/$(date +%Y)
```

---

### Solución 3: Verificar Configuración PHP

Editar `php.ini`:

```ini
upload_max_filesize = 10M
post_max_size = 10M
max_execution_time = 300
memory_limit = 256M
```

Reiniciar Apache:
```bash
sudo service apache2 restart
# o
sudo systemctl restart apache2
```

---

### Solución 4: Activar Logs de Errores

Editar `config/config.php`:

```php
// Cambiar estas líneas:
error_reporting(E_ALL);
ini_set('display_errors', 1);  // Cambiar de 0 a 1
```

Luego revisar:
```bash
tail -f logs/php_errors.log
# o
tail -f /var/log/apache2/error.log
```

---

### Solución 5: Verificar la API

Probar directamente la API desde navegador:

```
http://tu-dominio.com/public/api/certificados.php?action=generar
```

Si muestra error 404, verificar `.htaccess` y mod_rewrite:

```bash
sudo a2enmod rewrite
sudo service apache2 restart
```

---

### Solución 6: Usar Versión Simplificada (Sin Dependencias)

El sistema ahora incluye una versión simplificada que funciona sin TCPDF ni Dompdf.

**Características:**
- ✅ Genera PDF básico sin librerías externas
- ✅ Guarda HTML como respaldo
- ✅ Funciona inmediatamente

**Ya está implementado** en la última versión del archivo `CertificadoPDF.php`.

---

## 🔍 DIAGNÓSTICO PASO A PASO

### Paso 1: Verificar que la API responde

Abrir: `http://tu-dominio.com/public/api/certificados.php`

**Resultado esperado:**
```json
{
  "success": false,
  "message": "Acción no válida"
}
```

Si aparece error 404 o 500, hay un problema de configuración del servidor.

### Paso 2: Probar generación con cURL

```bash
curl -X POST "http://tu-dominio.com/public/api/certificados.php?action=generar" \
  -d "solicitud_id=1" \
  -H "Cookie: CERTRES_SESSION=tu_session_id"
```

### Paso 3: Revisar Logs

```bash
# Log de PHP
tail -f logs/php_errors.log

# Log de Apache
tail -f /var/log/apache2/error.log

# Log de MySQL
tail -f /var/log/mysql/error.log
```

### Paso 4: Verificar Base de Datos

```sql
-- Verificar tabla certificados
DESCRIBE certificados;

-- Verificar permisos de escritura
SHOW GRANTS;
```

---

## 🛠️ SCRIPT DE DIAGNÓSTICO AUTOMÁTICO

Crear archivo `test_certificado.php` en la raíz:

```php
<?php
require_once 'config/config.php';

echo "<h1>Test de Generación de Certificados</h1>";

// Test 1: Verificar directorios
echo "<h3>Test 1: Directorios</h3>";
$dirs = [
    CERTIFICADOS_PATH,
    CERTIFICADOS_PATH . '/' . date('Y'),
    CERTIFICADOS_PATH . '/' . date('Y') . '/' . date('m')
];

foreach ($dirs as $dir) {
    if (is_dir($dir)) {
        $writable = is_writable($dir);
        echo "✓ $dir - " . ($writable ? "<span style='color:green'>Escribible</span>" : "<span style='color:red'>No escribible</span>") . "<br>";
    } else {
        echo "✗ $dir - <span style='color:red'>No existe</span><br>";
        mkdir($dir, 0777, true);
        echo "→ Creado<br>";
    }
}

// Test 2: Verificar librerías
echo "<h3>Test 2: Librerías PDF</h3>";
$libs = ['TCPDF', '\Dompdf\Dompdf', '\Mpdf\Mpdf'];
foreach ($libs as $lib) {
    echo class_exists($lib) ? "✓ $lib disponible<br>" : "✗ $lib no disponible<br>";
}

// Test 3: Verificar extensiones PHP
echo "<h3>Test 3: Extensiones PHP</h3>";
$exts = ['pdo_mysql', 'mbstring', 'gd', 'zip'];
foreach ($exts as $ext) {
    echo extension_loaded($ext) ? "✓ $ext<br>" : "✗ $ext<br>";
}

// Test 4: Probar generación
echo "<h3>Test 4: Generar Certificado de Prueba</h3>";
try {
    $auth = new Auth();
    if (!$auth->isLoggedIn()) {
        echo "<span style='color:orange'>⚠ No autenticado - inicie sesión primero</span><br>";
    } else {
        $cert = new CertificadoPDF();
        $result = $cert->generar(1); // ID 1 de prueba
        
        if ($result['success']) {
            echo "<span style='color:green'>✓ Certificado generado: " . $result['numero_certificado'] . "</span><br>";
            echo "Ruta: " . $result['ruta_pdf'] . "<br>";
        } else {
            echo "<span style='color:red'>✗ Error: " . $result['message'] . "</span><br>";
        }
    }
} catch (Exception $e) {
    echo "<span style='color:red'>✗ Excepción: " . $e->getMessage() . "</span><br>";
}

echo "<hr>";
echo "<h3>Información del Sistema</h3>";
echo "PHP Version: " . phpversion() . "<br>";
echo "Memory Limit: " . ini_get('memory_limit') . "<br>";
echo "Max Execution Time: " . ini_get('max_execution_time') . "<br>";
echo "Upload Max Filesize: " . ini_get('upload_max_filesize') . "<br>";
```

Ejecutar: `http://tu-dominio.com/test_certificado.php`

---

## 📋 CHECKLIST DE VERIFICACIÓN

- [ ] Composer instalado
- [ ] Dependencias instaladas (`composer install`)
- [ ] Directorio `uploads/certificados/` existe
- [ ] Permisos 777 en `uploads/`
- [ ] PHP 8.0 o superior
- [ ] Extensiones PHP instaladas (pdo_mysql, mbstring)
- [ ] Logs activados
- [ ] Apache mod_rewrite habilitado
- [ ] Sesión activa (login exitoso)
- [ ] Base de datos conectada
- [ ] Tabla `certificados` existe

---

## 🎯 SOLUCIÓN RÁPIDA

Si tienes prisa y solo necesitas que funcione:

```bash
# 1. Permisos
chmod -R 777 uploads/

# 2. Crear directorios
mkdir -p uploads/certificados/$(date +%Y)/$(date +%m)

# 3. Verificar archivo actualizado
# El nuevo CertificadoPDF.php ya funciona sin dependencias

# 4. Probar
# Login → Dashboard → Ver Solicitud → Generar Certificado
```

---

## 📞 ERRORES COMUNES Y SOLUCIONES

### Error: "Class 'TCPDF' not found"
**Solución:** `composer require tecnickcom/tcpdf`

### Error: "Permission denied"
**Solución:** `chmod -R 777 uploads/`

### Error: "Failed to open stream"
**Solución:** Crear directorios manualmente

### Error: "Maximum execution time exceeded"
**Solución:** Aumentar `max_execution_time` en `php.ini`

### Error: "Allowed memory size exhausted"
**Solución:** Aumentar `memory_limit` en `php.ini`

---

## ✅ RESULTADO ESPERADO

Después de aplicar las soluciones:

1. Click en "Generar Certificado"
2. Mensaje: "Generando certificado..."
3. Mensaje: "✓ Certificado generado exitosamente"
4. Página se recarga
5. Aparece el certificado en la lista
6. Botones de: Descargar, Firmar, Enviar

---

**Última actualización:** 27 de Noviembre, 2025  
**Versión:** 1.0.3 (Error Fixed)
