# 📱 Sistema de Verificación con Código QR

## ✨ NUEVA FUNCIONALIDAD

El sistema ahora incluye:
1. **Firma profesional** con logo, como empresas chilenas
2. **Código QR** para verificación rápida
3. **Página de verificación pública** accesible a todos

---

## 🎨 DISEÑO DE LA FIRMA

### Basado en estándar chileno (como la imagen)

```
┌──────────────────────────────────────────────────────┐
│                                                       │
│  ┌───┐  FIRMADO DIGITALMENTE                         │
│  │ FD│  27/12/2025 15:30:45                          │
│  └───┘                                                │
│                                                       │
│  Nombre:        Juan Carlos Pérez Soto              │
│  Email:         jperez@municipio.cl                 │
│  Fecha Firma:   27/12/2025 15:30:45                 │
│  Válido hasta:  27/12/2026                          │
│                                          ┌─────────┐ │
│  Verifique la validez de este           │  █▀▀█  │ │
│  documento escaneando el código QR  →   │  █  █  │ │
│                                          │  ▀▀▀▀  │ │
│                                          └─────────┘ │
│ ───────────────────────────────────────────────────  │
│  Token: A3F5E89B2C4D1F7E                            │
│  Hash: a3f5e89b2c4d1f7e...1a2b3c4d5e6f7a8b         │
└──────────────────────────────────────────────────────┘
```

### Elementos Incluidos:

1. **Logo Circular** (superior izquierdo)
   - Círculo con borde bicolor (rojo/azul)
   - Iniciales del firmante
   - Tamaño: 20mm

2. **Título y Fecha**
   - "FIRMADO DIGITALMENTE" en negrita
   - Fecha/hora de firma

3. **Datos del Firmante**
   - Nombre completo
   - Email
   - Fecha de firma
   - Validez del certificado

4. **Código QR** (derecha)
   - Tamaño: 30mm × 30mm
   - Contiene URL de verificación
   - Nivel de corrección: L (Low)

5. **Información de Verificación**
   - Token único de 16 caracteres
   - Hash SHA-256 del documento

---

## 📱 CÓDIGO QR

### ¿Qué Contiene?

El QR incluye una URL como:
```
https://tu-dominio.com/public/verificar.php?cert=CERT-2025-00001&hash=a3f5e89b2c4d1f7e
```

### Parámetros de la URL:

- **cert**: Número de certificado
- **hash**: Primeros 16 caracteres del hash SHA-256

### Al Escanear:

1. Usuario escanea QR con su teléfono
2. Se abre página de verificación
3. Sistema busca el certificado
4. Valida el hash
5. Muestra resultado:
   - ✅ **Válido** - Verde con información completa
   - ⚠️ **Hash no coincide** - Advertencia
   - ❌ **No encontrado** - Certificado no existe

---

## 🌐 PÁGINA DE VERIFICACIÓN

### URL Pública:
```
http://tu-dominio.com/public/verificar.php
```

### Características:

1. **Búsqueda Manual**
   - Campo para número de certificado
   - Campo opcional para hash
   - Botón "Verificar Certificado"

2. **Resultados Detallados**
   
   **Si es válido:**
   - ✅ Check verde grande
   - "Certificado Válido y Auténtico"
   - Información completa:
     - Número de certificado
     - Fecha de emisión
     - Generado por (funcionario)
     - Estado de firma digital
     - Beneficiario
     - RUT
     - Dirección
     - Hash completo

   **Si hash no coincide:**
   - ⚠️ Advertencia en rojo
   - "Hash No Coincide"
   - Posibles razones
   - Recomendación de contacto

   **Si no existe:**
   - ❌ X amarilla
   - "Certificado No Encontrado"
   - Posibles razones
   - Sugerencias

3. **Diseño Responsive**
   - Se ve bien en móviles
   - Gradiente profesional
   - Iconos Font Awesome

---

## 🔒 SEGURIDAD

### Validaciones:

1. **Número de Certificado**
   - Formato: CERT-YYYY-NNNNN
   - Único en BD

2. **Hash SHA-256**
   - Del documento PDF final
   - 64 caracteres hexadecimales
   - Primeros 16 en QR

3. **Token de Verificación**
   - Generado al crear certificado
   - Basado en cert + hash + timestamp
   - Alfanumérico, 16 caracteres

### Protección contra Fraude:

- ✅ Hash detecta modificaciones al PDF
- ✅ Token único por documento
- ✅ Fecha de emisión verificable
- ✅ Firma digital con certificado válido
- ✅ Registro completo en BD

---

## 📖 CÓMO USAR (Usuario Final)

### Verificar Certificado Escaneando QR:

1. Recibir certificado (PDF o impreso)
2. Buscar código QR (esquina inferior izquierda)
3. Abrir cámara del teléfono
4. Escanear QR
5. ✅ Ver resultado de verificación

### Verificar Certificado Manualmente:

1. Ir a: `tu-dominio.com/public/verificar.php`
2. Ingresar número de certificado (ej: CERT-2025-00001)
3. Opcional: Ingresar primeros 16 del hash
4. Click "Verificar Certificado"
5. ✅ Ver resultado

---

## 💼 CÓMO FUNCIONA (Técnico)

### Generación del QR:

```php
// Durante la generación del PDF
$urlVerificacion = $this->generarURLVerificacion($certificado);

// TCPDF genera QR
$pdf->write2DBarcode(
    $urlVerificacion,
    'QRCODE,L',  // QRCODE con error correction Low
    $qrX,        // Posición X
    $qrY,        // Posición Y
    30,          // Ancho en mm
    30,          // Alto en mm
    array(       // Opciones
        'border' => false,
        'fgcolor' => array(0, 0, 0),      // Negro
        'bgcolor' => array(255, 255, 255) // Blanco
    )
);
```

### URL de Verificación:

```php
function generarURLVerificacion($certificado) {
    $baseUrl = $_SERVER['HTTP_HOST'];
    $protocolo = isset($_SERVER['HTTPS']) ? 'https' : 'http';
    
    $url = $protocolo . '://' . $baseUrl . 
           '/public/verificar.php?cert=' . 
           urlencode($certificado['numero_certificado']) . 
           '&hash=' . 
           substr($certificado['hash_documento'], 0, 16);
    
    return $url;
}
```

### Verificación:

```php
// En verificar.php
$cert = $_GET['cert'];
$hash = $_GET['hash'];

// Buscar en BD
$certificado = $db->selectOne(
    "SELECT * FROM certificados WHERE numero_certificado = ?",
    [$cert]
);

// Validar hash
if ($certificado) {
    $hashReal = substr($certificado['hash_documento'], 0, 16);
    
    if ($hash === $hashReal) {
        // ✅ Válido
    } else {
        // ⚠️ Hash no coincide
    }
}
```

---

## 🎨 PERSONALIZACIÓN

### Cambiar Colores del Logo:

En `agregarFirmaVisualTCPDF()`:

```php
// Logo círculo
$pdf->SetFillColor(41, 128, 185);  // RGB del relleno
$pdf->SetDrawColor(231, 76, 60);   // RGB del borde
```

### Cambiar Tamaño del QR:

```php
$qrSize = 30; // 30mm (ajustar según necesidad)
```

### Cambiar Posición de la Firma:

```php
$pdf->SetY(-75);  // 75mm desde el fondo (ajustar)
$startX = 20;     // 20mm desde la izquierda
$anchoTotal = 170; // Ancho del cuadro
```

---

## 📊 ESTADÍSTICAS

### Datos Almacenados:

Por cada certificado:
- ✅ Número único
- ✅ Hash SHA-256
- ✅ Token de verificación
- ✅ Fecha de generación
- ✅ Fecha de firma (si aplica)
- ✅ Usuario que generó
- ✅ Solicitante (nombre, RUT)

### Consultas Posibles:

```sql
-- Total de verificaciones (agregar tabla logs)
SELECT COUNT(*) FROM logs_verificacion;

-- Certificados más verificados
SELECT numero_certificado, COUNT(*) as verificaciones
FROM logs_verificacion
GROUP BY numero_certificado
ORDER BY verificaciones DESC;
```

---

## 🔍 CASOS DE USO

### Caso 1: Ciudadano Verifica Certificado

```
María recibe un certificado de residencia por email

1. Abre el PDF
2. Ve el código QR
3. Abre cámara del celular
4. Escanea QR
5. Navegador abre página de verificación
6. ✅ Ve "Certificado Válido y Auténtico"
7. Confirma que es legítimo
```

### Caso 2: Empresa Verifica Certificado

```
Banco necesita verificar certificado de residencia

1. Cliente presenta certificado impreso
2. Ejecutivo ve código QR
3. Escanea con tablet de la empresa
4. Sistema muestra:
   - ✅ Válido
   - Nombre coincide
   - RUT coincide
   - Dirección verificada
5. Aprueba solicitud
```

### Caso 3: Detección de Fraude

```
Se presenta certificado adulterado

1. Certificado tiene datos modificados
2. Usuario escanea QR
3. Sistema calcula hash del PDF actual
4. ⚠️ Hash no coincide con el registrado
5. Muestra advertencia
6. Recomienda contactar municipalidad
```

---

## ⚠️ IMPORTANTE

### Requisitos:

1. **TCPDF debe estar instalado**
   ```bash
   composer require tecnickcom/tcpdf
   ```

2. **Extensión GD de PHP**
   ```bash
   php -m | grep gd
   ```

3. **Base de datos actualizada**
   - Tabla `certificados` con campo `hash_documento`

### Limitaciones:

- QR funciona con conexión a internet
- URL debe ser accesible públicamente
- Hash solo detecta cambios en archivo, no validez legal

---

## ✅ CHECKLIST DE IMPLEMENTACIÓN

- [x] Firma profesional con logo
- [x] Código QR integrado
- [x] URL de verificación generada
- [x] Página de verificación pública
- [x] Validación de hash
- [x] Token de seguridad
- [x] Diseño responsive
- [x] Información completa del certificado
- [x] Manejo de errores
- [x] Mensajes claros para usuarios

---

## 📱 RESULTADO FINAL

### En el PDF:

```
┌─────────────────────────────────────┐
│  ⭕ FD  FIRMADO DIGITALMENTE        │
│         27/12/2025 15:30            │
│                                      │
│  Nombre:     Juan Pérez              │
│  Email:      jperez@correo.cl       │
│  Firmado:    27/12/2025 15:30       │
│  Válido:     27/12/2026             │
│                                      │
│  Verifique escaneando →  [QR CODE] │
│                                      │
│  Token: A3F5E89B2C4D1F7E            │
│  Hash:  a3f5e89b...                 │
└─────────────────────────────────────┘
```

### Al Escanear QR:

Usuario ve página con:
- ✅ Grande check verde
- "Certificado Válido y Auténtico"
- Toda la información del certificado
- Opción de verificar otro

---

**Versión:** 1.0.6 (QR Verification)  
**Fecha:** 27 de Diciembre, 2025  
**Estado:** ✅ Sistema Completo con QR
